import TestDynamicIpAssignmentSettings from "./TestDynamicIpAssignmentSettings.js";


export default {
    template: `
        <div>
            <test-dynamic-ip-assignment-settings ref="testdynamicIpAssignmentSettings" :xhr-url="xhrUrl"></test-dynamic-ip-assignment-settings>

            <div class="modal fade" ref="ipConfigHelpModal" role="dialog" @click.self="closeConfigurableOptionDocumentation()" tabindex="0">
                <div class="modal-dialog modal-lg" role="document">
                    <div class='modal-content panel panel-primary'>
                        <div class='modal-header panel-heading'>
                            <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                <span aria-hidden="true">&times;</span>
                            </button>
                            <h4 class="modal-title">Configurable Options: Dynamic IP Assignments</h4>
                        </div>
                        <div class='modal-body panel-body'>
                            <div class="alert dservermanager-alert-secondary">
                                <div>
                                    While the dynamic IP assignment system may appear complex at first glance, it's designed to provide maximum flexibility while remaining easy to implement. It is recommended to use the testing tool ("Test configuration with a service") to validate your configurations - this will help ensure your IP assignment settings work exactly as intended before deployment.
                                </div>
                            </div>

                            <div class="panel-group" id="ipConfigAccordion">
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4 class="panel-title">
                                            <a data-toggle="collapse" data-parent="#ipConfigAccordion" href="#implementation">
                                                Implementation Guide
                                            </a>
                                        </h4>
                                    </div>
                                    <div id="implementation" class="panel-collapse collapse in">
                                        <div class="panel-body">
                                            <b>Setting Up Configurable Options:</b>
                                            <ol>
                                                <li>
                                                    <a href="configproductoptions.php" target="_blank">Create a configurable option group and create options for IPv4 and IPv6 assignments.</a>
                                                    <br/>
                                                    You can create either IPv4 assignments options, IPv6 assignments options, or both - depending on your needs.
                                                </li>
                                                <li>Use option names:
                                                    <ul>
                                                        <li>
                                                            <code>dynamicIPv4Assignments</code>
                                                        </li>
                                                        <li>
                                                            <code>dynamicIPv6Assignments</code>
                                                        </li>
                                                    </ul>
                                                </li>
                                                <li>Create custom options using the configuration format below.</li>
                                            </ol>

                                            <b>Important Notes:</b>
                                            <ul>
                                                <li>Configurable options override both global and product-specific settings.</li>
                                                <li>If a parameter is not set in the configurable option, the system will use:
                                                    <ol>
                                                        <li>First, the product-specific setting if configured.</li>
                                                        <li>Then, the global system default if no product setting exists.</li>
                                                    </ol>
                                                </li>
                                                <li>This applies to all parameters except:
                                                    <ul>
                                                        <li>Assignment Type (
                                                            <code>type=subnet/address</code>
                                                            )
                                                        </li>
                                                    </ul>
                                                </li>
                                            </ul>

                                            <b>Example Setup (Friendly Display Names are supported):</b>
                                            <pre style="margin-top: 5px;">Option Name: dynamicIPv4Assignments|IPv4 Addresses
                                                
Options:
- type=address@count=1|1 IPv4 Address
- type=subnet@count=1@cidrSize=29|/29 Subnet
- none|No IP Assignment
- count= OR count=0|No IP Assignment
(see below for information about the difference between 'none' and 'count=0')</pre>
                                            In this example,
                                            <a href="https://docs.whmcs.com/products/configuration-options/configurable-options/#friendly-display-names" target="_blank">"Friendly Display Names"</a>
                                            have been used.

                                            <div style="margin-top: 10px">
                                                <b>Required Settings</b>

                                                <div style="margin-top: 5px;">
                                                    The following parameters
                                                    <b>must be set</b>
                                                    for subnet assignments:
                                                    <table class="table table-bordered" style="margin-top: 5px;">
                                                        <thead>
                                                        <tr>
                                                            <th>Parameter</th>
                                                            <th>Example</th>
                                                            <th>Inheritance</th>
                                                        </tr>
                                                        </thead>

                                                        <tbody>
                                                        <tr>
                                                            <td>
                                                                <code>type</code>
                                                            </td>
                                                            <td>type=subnet</td>
                                                            <td>Cannot be inherited from global or product configuration (must exist in config option)</td>
                                                        </tr>
                                                        <tr>
                                                            <td>
                                                                <code>count</code>
                                                            </td>
                                                            <td>count=1</td>
                                                            <td>Can be inherited from global or product configuration</td>
                                                        </tr>
                                                        <tr>
                                                            <td>
                                                                <code>cidrSize</code>
                                                            </td>
                                                            <td>cidrSize=24</td>
                                                            <td>Can be inherited from global or product configuration</td>
                                                        </tr>
                                                        </tbody>
                                                    </table>

                                                    The following parameters
                                                    <b>must be set</b>
                                                    for single IP assignments:
                                                    <table class="table table-bordered" style="margin-top: 5px;">
                                                        <thead>
                                                        <tr>
                                                            <th>Parameter</th>
                                                            <th>Example</th>
                                                            <th>Inheritance</th>
                                                        </tr>
                                                        </thead>
                                                        <tbody>
                                                        <tr>
                                                            <td>
                                                                <code>type</code>
                                                            </td>
                                                            <td>type=address</td>
                                                            <td>Cannot be inherited from global or product configuration (must exist in config option)</td>
                                                        </tr>
                                                        <tr>
                                                            <td>
                                                                <code>count</code>
                                                            </td>
                                                            <td>count=1</td>
                                                            <td>Can be inherited from global or product configuration</td>
                                                        </tr>
                                                        </tbody>
                                                    </table>

                                                    <b>Important:</b>
                                                    If the
                                                    <code>count</code>
                                                    parameter is not defined anywhere, the count is treated as
                                                    <code>0</code>
                                                    and the system will not assign any IP addresses for the respective protocol.
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Basic Format -->
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4 class="panel-title">
                                            <a data-toggle="collapse" data-parent="#ipConfigAccordion" href="#basicFormat">
                                                Basic Configuration Format
                                            </a>
                                        </h4>
                                    </div>
                                    <div id="basicFormat" class="panel-collapse collapse">
                                        <div class="panel-body">
                                            <div>
                                                <b>Configurations use a key-value format separated by @ symbols:</b>
                                            </div>
                                            <pre style="margin-top: 5px;">type=subnet@cidrSize=29@count=1@subnetSelectionMode=priorityFirst</pre>
                                            <div>
                                                <b>Use 'none' to disable IP assignments:</b>
                                            </div>
                                            <pre style="margin-top: 5px;">none</pre>
                                            <div>Note: When using the 'none' option, no IP assignment will be done for the respective protocol. However, if you assign an IP of the other protocol (e.g., IPv4 while IPv6 is set to 'none'), the system will still respect the "Delete Existing IP Assignments" setting and may remove existing IPs (see "Common Configuration Examples" for more information).</div>

                                            <div style="margin-top: 10px;">
                                                <b>To unset a default parameter (e.g., CIDR size), use an empty value:</b>
                                                <pre style="margin-top: 5px;">type=address@cidrSize=@restrictToSameSubnet=@count=1</pre>
                                            </div>

                                            <div style="margin-top: 10px;">
                                                <b>
                                                    <a href="https://docs.whmcs.com/products/configuration-options/configurable-options/#friendly-display-names" target="_blank">"Friendly Display Names"</a>
                                                    are supported:
                                                </b>
                                            </div>
                                            <pre style="margin-top: 5px;">Example: type=subnet@cidrSize=29@count=1@subnetSelectionMode=priorityFirst|/29 Subnet
https://docs.whmcs.com/products/configuration-options/configurable-options/#friendly-display-names</pre>
                                        </div>
                                    </div>
                                </div>

                                <!-- Common Examples -->
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4 class="panel-title">
                                            <a data-toggle="collapse" data-parent="#ipConfigAccordion" href="#commonExamples">
                                                Common Configuration Examples
                                            </a>
                                        </h4>
                                    </div>
                                    <div id="commonExamples" class="panel-collapse collapse">
                                        <div class="panel-body">
                                            <b>Single IP Addresses:</b>
                                            <pre style="margin-top: 5px;">type=address@count=1</pre>
                                            <div>This assigns a single IP address using the CIDR size of the global/product configuration (if set).</div>

                                            <div style="margin-top: 10px;">
                                                <b>Multiple IP Addresses:</b>
                                                <pre style="margin-top: 5px;">type=address@count=5@cidrSize=24@assignmentMode=preferSequentialIps</pre>
                                                <div>This assigns 5 IP addresses from a /24 network and the assignment mode is set to "preferSequentialIps".</div>
                                            </div>

                                            <div style="margin-top: 10px;">
                                                <b>Subnet Assignment:</b>
                                                <pre style="margin-top: 5px;">type=subnet@cidrSize=29@count=1</pre>
                                                <div>This assigns a single /29 subnet.</div>
                                            </div>

                                            <div style="margin-top: 10px;">
                                                <b>Multiple Subnets:</b>
                                                <pre style="margin-top: 5px;">type=subnet@cidrSize=30@count=2</pre>
                                                <div>This assigns two /30 subnets.</div>
                                            </div>

                                            <div style="margin-top: 10px;">
                                                <b>No Automatic Assignment (relevant for IPv6):</b>
                                                <pre style="margin-top: 5px;">type=subnet@cidrSize=30@count=0</pre>
                                                <div>This will remove all IPv6 subnet assignments (the cidrSize is a required parameter, but ignored. If the CIDR size is set in the global or product configuration, it must not be set in the configurable option because it is inherited).</div>
                                                <div style="margin-top: 5px;">
                                                    Note: The 'none' option and 'count=0' option differ in how they handle API communication. With 'none', no API payload is sent for that protocol, and if both IPv4 and IPv6 are set to 'none', no API call occurs at all. In contrast, 'count=0' actively sends a command to remove existing IPv4 or IPv6 assignments.
                                                </div>
                                                <div style="margin-top: 5px;">
                                                    While both options often achieve similar results, it is strongly recommended to make tests to fully understand the assignment options. The testing tool makes it easy to validate your configurations.
                                                </div>
                                            </div>

                                            <div style="margin-top: 10px;">
                                                <b>Unsetting a default parameter:</b>
                                                <pre style="margin-top: 5px;">type=address@cidrSize=</pre>
                                                <div>Since no count parameter is set, the number of assigned single IP addresses follows the global or product configuration. And since the CIDR size is unset, it removes any CIDR size restrictions from the global/product configuration (if set) and the system will assign an IP from any available subnet size.</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Available Parameters -->
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4 class="panel-title">
                                            <a data-toggle="collapse" data-parent="#ipConfigAccordion" href="#parameters">
                                                Available Parameters
                                            </a>
                                        </h4>
                                    </div>
                                    <div id="parameters" class="panel-collapse collapse">
                                        <div class="panel-body">
                                            The order of parameters does not matter. The following parameters are available:
                                            <br/>
                                            <br/>

                                            <b>Common Parameters (Required):</b>
                                            <ul>
                                                <li>
                                                    <code>type</code>
                                                    : Either 'address' for single IPs or 'subnet' for subnet assignments
                                                </li>
                                            </ul>

                                            <b>Single IP Assignment Parameters:</b>
                                            <ul>
                                                <li>
                                                    <code>count</code>
                                                    : Number of IPs to assign
                                                </li>
                                                <li>
                                                    <code>cidrSize</code>
                                                    : Network size the IPs should come from
                                                </li>
                                                <li>
                                                    <code>assignmentMode</code>
                                                    : How IPs should be assigned
                                                    <ul>
                                                        <li>
                                                            <code>preferSequentialIps</code>
                                                            : Try sequential IPs first, fall back to non-sequential if needed
                                                        </li>
                                                        <li>
                                                            <code>failIfNotSequentialIps</code>
                                                            : Only assign if sequential IPs are available
                                                        </li>
                                                        <li>
                                                            <code>allowNonSequentialIps</code>
                                                            : Allow non-sequential IP assignments
                                                        </li>
                                                    </ul>
                                                </li>
                                                <li>
                                                    <code>restrictToSameSubnet</code>
                                                    : Whether all IPs must come from the same subnet (true/false)
                                                </li>
                                            </ul>

                                            <b>Subnet Assignment Parameters:</b>
                                            <ul>
                                                <li>
                                                    <code>count</code>
                                                    : Number of subnets to assign
                                                </li>
                                                <li>
                                                    <code>cidrSize</code>
                                                    : Size of the subnet to assign
                                                    <ul>
                                                        <li>For IPv4: 1-32</li>
                                                        <li>For IPv6: 1-128</li>
                                                    </ul>
                                                </li>
                                                <li>
                                                    <code>assignmentMode</code>
                                                    : How subnets should be assigned
                                                    <ul>
                                                        <li>
                                                            <code>preferExistingOrSplitSubnet</code>
                                                            : Try existing subnets first, create new if needed
                                                        </li>
                                                        <li>
                                                            <code>requireExistingSubnet</code>
                                                            : Only use existing subnets
                                                        </li>
                                                        <li>
                                                            <code>requireSplitSubnet</code>
                                                            : Always create new subnets
                                                        </li>
                                                    </ul>
                                                </li>
                                                <li>
                                                    <code>subnetSelectionMode</code>
                                                    : How to select parent subnets
                                                    <ul>
                                                        <li>
                                                            <code>priorityFirst</code>
                                                            : Select based on subnet priority
                                                        </li>
                                                        <li>
                                                            <code>smallestSubnetFirst</code>
                                                            : Prefer smaller subnets
                                                        </li>
                                                        <li>
                                                            <code>largestSubnetFirst</code>
                                                            : Prefer larger subnets
                                                        </li>
                                                    </ul>
                                                </li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4 class="panel-title">
                                            <a data-toggle="collapse" data-parent="#ipConfigAccordion" href="#events">
                                                Programmatic Events
                                            </a>
                                        </h4>
                                    </div>
                                    <div id="events" class="panel-collapse collapse">
                                        <div class="panel-body">
                                            <div>You can programmatically modify the IP assignment payload using the event system.</div>

                                            <div style="margin-top: 10px;">
                                                <b>Event name:</b>
                                                <code>overrideDynamicIpAssignmentPayload</code>
                                            </div>

                                            <div style="margin-top: 10px;">
                                                <b>Event Parameters:</b>
                                                <ul>
                                                    <li>
                                                        <code>$serviceDetails</code>
                                                        : WHMCS service details array
                                                    </li>
                                                    <li>
                                                        <code>$payload</code>
                                                        : The IP assignment payload that can be modified
                                                    </li>
                                                </ul>
                                            </div>

                                            <div style="margin-top: 10px;">
                                                <b>The event allows you to:</b>
                                                <ul>
                                                    <li>Modify the IP assignment payload before it's processed.</li>
                                                    <li>Return 'skip' to cancel the IP assignment.</li>
                                                    <li>Return a modified payload to change the assignment behavior.</li>
                                                </ul>
                                            </div>

                                            <div style="margin-top: 10px;">
                                                <b>Event location:</b>
                                                <pre>/modules/addons/dservermanager/app/Custom/events.example</pre>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                        </div>
                    </div>
                </div>
            </div>


            <div class="alert dservermanager-alert-secondary">
                <div>
                    This system is able to automatically assign IP addresses to servers based on a hierarchical configuration:
                </div>

                <ul style="margin-top: 5px; margin-bottom: 5px;">
                    <li>Global Settings
                        <span v-if="mode !== 'productSetting'">(current view)</span>
                        - Define the default behavior for all products
                    </li>
                    <li>WHMCS Product Settings
                        <span v-if="mode === 'productSetting'">(current view)</span>
                        - Can override global settings for specific products in WHMCS
                    </li>
                    <li>
                        Configurable Options - Allow customers to customize IP assignments during ordering (e.g., number of IPs, subnet sizes)
                    </li>
                </ul>

                <div>
                    If the assignment options are disabled, the servers will keep the currently assigned IP addresses.
                </div>

                <div style="margin-top: 5px;">
                    <button class="btn btn-default" @click.prevent="$refs.testdynamicIpAssignmentSettings.openModal()">
                        <i class="far fa-cogs"></i>
                        Test configuration with a service
                    </button>

                    <button style="margin-left: 5px;" class="btn btn-default" @click.prevent="openConfigurableOptionDocumentation">
                        <i class="far fa-info-circle"></i>
                        Configurable Options Documentation
                    </button>
                </div>
            </div>

            <div style="padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Assign IPv4</b>
                        <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip"
                           title="Specify the number of addresses and whether they should be assigned as individual IPs or as a subnet. Existing IP assignments may be removed based on deletion settings."></i>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_assign_ipv4_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_assign_ipv4_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_assign_ipv4_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_assign_ipv4_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <div class="row">
                        <div class="col-md-2">
                            <label class="control-label">Enable
                                <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip"
                                   title="Enables or disables automatic IP assignments. Like all settings, this can be overridden on a per-product basis or via configurable options."></i>
                            </label>
                            <div>
                                <button :disabled="isDisabled('dynamicIpAssignments_assign_ipv4_productSettingMode')" type="button" class="btn btn-sm"
                                        @click="settingValues.dynamicIpAssignments_assign_ipv4_enable = { yes: 'no', no: 'yes' }[settingValues.dynamicIpAssignments_assign_ipv4_enable]"
                                        :class="{'btn-primary': settingValues.dynamicIpAssignments_assign_ipv4_enable === 'yes'}">
                                    {{ settingValues.dynamicIpAssignments_assign_ipv4_enable === 'yes' ? 'Enabled' : 'Disabled' }}
                                </button>
                            </div>
                        </div>

                        <div class="col-md-3">
                            <label class="control-label">Count</label>
                            <input :disabled="isDisabled('dynamicIpAssignments_assign_ipv4_productSettingMode')" type="number" class="form-control" v-model="settingValues.dynamicIpAssignments_assign_ipv4_count" placeholder="Count" min="1">
                        </div>

                        <div class="col-md-4">
                            <label class="control-label">Assignment Type</label>
                            <select :disabled="isDisabled('dynamicIpAssignments_assign_ipv4_productSettingMode')" class="form-control" v-model="settingValues.dynamicIpAssignments_assign_ipv4_type">
                                <option value="single">Single IPs</option>
                                <option value="subnet">Subnet</option>
                            </select>
                        </div>

                        <div class="col-md-3">
                            <label class="control-label">CIDR
                                <span v-if="settingValues.dynamicIpAssignments_assign_ipv4_type !== 'subnet'">(Optional)</span>
                                <span v-else>(Required)</span>
                            </label>
                            <input :disabled="isDisabled('dynamicIpAssignments_assign_ipv4_productSettingMode')" type="number" class="form-control" v-model="settingValues.dynamicIpAssignments_assign_ipv4_cidrSize" placeholder="CIDR Size" min="1"
                                   max="32">
                        </div>
                    </div>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Assign IPv6</b>
                        <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip"
                           title="Specify the number of addresses and whether they should be assigned as individual IPs or as a subnet. Existing IP assignments may be removed based on deletion settings."></i>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_assign_ipv6_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_assign_ipv6_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_assign_ipv6_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_assign_ipv6_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <div class="row">
                        <div class="col-md-2">
                            <label class="control-label">Enable
                                <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip"
                                   title="Enables or disables automatic IP assignments. Like all settings, this can be overridden on a per-product basis or via configurable options."></i>
                            </label>
                            <div>
                                <button :disabled="isDisabled('dynamicIpAssignments_assign_ipv6_productSettingMode')" type="button" class="btn btn-sm"
                                        @click="settingValues.dynamicIpAssignments_assign_ipv6_enable = { yes: 'no', no: 'yes' }[settingValues.dynamicIpAssignments_assign_ipv6_enable]"
                                        :class="{'btn-primary': settingValues.dynamicIpAssignments_assign_ipv6_enable === 'yes'}">
                                    {{ settingValues.dynamicIpAssignments_assign_ipv6_enable === 'yes' ? 'Enabled' : 'Disabled' }}
                                </button>
                            </div>
                        </div>

                        <div class="col-md-3">
                            <label class="control-label">Count</label>
                            <input :disabled="isDisabled('dynamicIpAssignments_assign_ipv6_productSettingMode')" type="number" class="form-control" v-model="settingValues.dynamicIpAssignments_assign_ipv6_count" placeholder="Count" min="1">
                        </div>

                        <div class="col-md-4">
                            <label class="control-label">Assignment Type</label>
                            <select :disabled="isDisabled('dynamicIpAssignments_assign_ipv6_productSettingMode')" class="form-control" v-model="settingValues.dynamicIpAssignments_assign_ipv6_type">
                                <option value="single">Single IPs</option>
                                <option value="subnet">Subnet</option>
                            </select>
                        </div>

                        <div class="col-md-3">
                            <label class="control-label">CIDR
                                <span v-if="settingValues.dynamicIpAssignments_assign_ipv6_type !== 'subnet'">(Optional)</span>
                                <span v-else>(Required)</span>
                            </label>
                            <input :disabled="isDisabled('dynamicIpAssignments_assign_ipv6_productSettingMode')" type="number" class="form-control" v-model="settingValues.dynamicIpAssignments_assign_ipv6_cidrSize" placeholder="CIDR Size" min="1"
                                   max="128">
                        </div>
                    </div>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Default IP Assignment Mode</b>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_ipAssignmentMode_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_ipAssignmentMode_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_ipAssignmentMode_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_ipAssignmentMode_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <select :disabled="isDisabled('dynamicIpAssignments_ipAssignmentMode_productSettingMode')" class="form-control" v-model="settingValues.dynamicIpAssignments_ipAssignmentMode">
                        <option v-for="option in [
                                                    { id: 'preferSequentialIps', name: 'Prefer Sequential IPs' },
                                                    { id: 'failIfNotSequentialIps', name: 'Require Sequential IPs' },
                                                    { id: 'allowNonSequentialIps', name: 'Allow Non-Sequential IPs' }
                                                ]" :value="option.id">{{ option.name }}
                        </option>
                    </select>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0;">
                    <div>
                        <b>Default Subnet Assignment Mode</b>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_subnetAssignmentMode_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_subnetAssignmentMode_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_subnetAssignmentMode_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_subnetAssignmentMode_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <select :disabled="isDisabled('dynamicIpAssignments_subnetAssignmentMode_productSettingMode')" class="form-control" v-model="settingValues.dynamicIpAssignments_subnetAssignmentMode">
                        <option v-for="option in [
                                                    { id: 'preferExistingOrSplitSubnet', name: 'Prefer Existing or Split Subnet' },
                                                    { id: 'requireExistingSubnet', name: 'Require Existing Subnet' },
                                                    { id: 'requireSplitSubnet', name: 'Require Split Subnet' }
                                                ]" :value="option.id">{{ option.name }}
                        </option>
                    </select>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Default Subnet Selection Mode</b>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_subnetSelectionMode_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_subnetSelectionMode_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_subnetSelectionMode_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_subnetSelectionMode_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <select :disabled="isDisabled('dynamicIpAssignments_subnetSelectionMode_productSettingMode')" class="form-control" v-model="settingValues.dynamicIpAssignments_subnetSelectionMode">
                        <option v-for="option in [
                                                    { id: 'priorityFirst', name: 'Priority First' },
                                                    { id: 'smallestSubnetFirst', name: 'Smallest Subnet First' },
                                                    { id: 'largestSubnetFirst', name: 'Largest Subnet First' }
                                                ]" :value="option.id">{{ option.name }}
                        </option>
                    </select>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Reserve IPs when splitting</b>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_reserveIps_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_reserveIps_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_reserveIps_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_reserveIps_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_reserveIps_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_reserveNetworkAddress" true-value="yes" false-value="no">
                            Reserve Network Address
                        </label>
                    </div>
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_reserveIps_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_reserveGatewayIp" true-value="yes" false-value="no">
                            Reserve Gateway IP
                        </label>
                    </div>
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_reserveIps_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_reserveBroadcastIp" true-value="yes" false-value="no">
                            Reserve Broadcast IP
                        </label>
                    </div>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Require IPs from same subnet</b>
                        <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip"
                           title="When enabled, all assigned IPs will be from the same subnet. This setting has no effect if you're using the 'Require Sequential IPs' mode (since sequential IPs are always from the same subnet) or if you're only assigning subnets."></i>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_restrictToSameSubnet_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_restrictToSameSubnet_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_restrictToSameSubnet_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_restrictToSameSubnet_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <div class="checkbox" style="margin-top: 0;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_restrictToSameSubnet_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_restrictToSameSubnet" true-value="yes" false-value="no">
                            Require all IPs to be from the same subnet
                        </label>
                    </div>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Reseller Handling</b>
                        <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip" title="These actions are only performed if the user is a reseller."></i>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_resellerHandling_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_resellerHandling_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_resellerHandling_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_resellerHandling_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_resellerHandling_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_useResellerSubnets" true-value="yes" false-value="no">
                            Assign IPs/Split subnets of reseller-owned subnets if user is a reseller
                            <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip"
                               title="When assigning IPs or creating new subnets, the system will use subnets owned by the reseller. Enable the option below to allow a fallback to your subnets."></i>
                        </label>
                    </div>
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_resellerHandling_productSettingMode') || settingValues.dynamicIpAssignments_useResellerSubnets !== 'yes'" type="checkbox"
                                   v-model="settingValues.dynamicIpAssignments_allowOwnSubnetsAsFallbackForRestrictedUser" true-value="yes" false-value="no">
                            Allow fallback to your owned subnets if no reseller subnets are available or matching
                            <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip"
                               title="When assigning IPs or creating new subnets, the system will first attempt to use subnets owned by the reseller. If no suitable reseller-owned subnets are available, one from your owned subnets will be used."></i>
                        </label>
                    </div>
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_resellerHandling_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_assignCreatedSubnetToReseller" true-value="yes" false-value="no">
                            When assigning a subnet to a server, assign the subnet to the reseller
                            <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip" title="When new subnets are created through splitting, they will be assigned to the reseller's ownership."></i>
                        </label>
                    </div>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12" v-if="can('servers.ipassignments.store.param.performVlanActions')">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Perform Network Automation Actions</b>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_vlan_productSettingMode = 'inherit'" :class="{'btn-primary': settingValues.dynamicIpAssignments_vlan_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_vlan_productSettingMode = 'override'" :class="{'btn-primary': settingValues.dynamicIpAssignments_vlan_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_vlan_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_vlan_afterAllIpAssignments" true-value="yes" false-value="no">
                            After all IP assignment actions (deletions and assignments) are completed.
                        </label>
                    </div>
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_vlan_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_vlan_afterDeleteExistingIpAssignments" true-value="yes" false-value="no">
                            After deleting existing IP assignments
                        </label>
                    </div>
                    <div class="checkbox" style="margin-top: 0px; margin-bottom: 0px;">
                        <label>
                            <input :disabled="isDisabled('dynamicIpAssignments_vlan_productSettingMode')" type="checkbox" v-model="settingValues.dynamicIpAssignments_vlan_afterEachIpAssignment" true-value="yes" false-value="no">
                            After each IP assignment
                        </label>
                    </div>
                </div>
            </div>

            <div style="padding-top: 25px; padding-left: 0px; padding-right: 0px;" class="col-md-12">
                <div class="col-md-4" style="padding-left: 0px; padding-right: 0px;">
                    <div>
                        <b>Delete Existing IP Assignments</b>
                        <i style="margin-left: 3px;" class="far fa-info-circle" data-toggle="dynamic-tooltip"
                           title="Controls whether existing IP assignments are removed before new assignments. Example: If using dynamic IPv4 assignments but manual IPv6 assignments, selecting 'Delete IPv4 Only' preserves manual IPv6 assignments."></i>
                    </div>

                    <div v-if="mode === 'productSetting'">
                        <div class="btn-group" style="margin-top: 5px;">
                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_deleteExistingIpAssignments_productSettingMode = 'inherit'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_deleteExistingIpAssignments_productSettingMode === 'inherit'}">
                                <i class="fa fa-link"></i>
                                Inherit
                            </button>

                            <button type="button" class="btn btn-xs" @click="settingValues.dynamicIpAssignments_deleteExistingIpAssignments_productSettingMode = 'override'"
                                    :class="{'btn-primary': settingValues.dynamicIpAssignments_deleteExistingIpAssignments_productSettingMode === 'override'}">
                                <i class="fa fa-unlink"></i>
                                Override
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-8">
                    <select :disabled="isDisabled('dynamicIpAssignments_deleteExistingIpAssignments_productSettingMode')" class="form-control" v-model="settingValues.dynamicIpAssignments_deleteExistingIpAssignments">
                        <option v-for="option in [
                                { id: 'ipv4', name: 'Delete IPv4 Only' },
                                { id: 'ipv6', name: 'Delete IPv6 Only' },
                                { id: 'both', name: 'Delete Both IPv4 and IPv6' }
                            ]" :value="option.id">{{ option.name }}
                        </option>
                    </select>
                </div>
            </div>

            <div class="col-md-12" style="padding-top: 25px;">
                <button v-if="mode === 'productSetting'" class="pull-right btn btn-primary" @click="saveProductSettings" :disabled="pendingXhrRequests">Save Changes</button>
            </div>
        </div>
    `,

    props: [
        'value',
        'mode',
        'productId',
        'xhrUrl',
        'can',
        'userPermissions'
    ],

    components: {
        TestDynamicIpAssignmentSettings
    },

    mounted() {
        if (this.mode === 'productSetting') {
            this.getProductSettings();
        }
    },

    data() {
        return {
            productSettingValues: {},
            pendingXhrRequests: false,
        }
    },


    computed: {
        settingValues: {
            get() {
                // Differenzierung anhand des Modus
                return this.mode === 'productSetting' ? this.productSettingValues : this.value;
            },
            set(val) {
                if (this.mode === 'productSetting') {
                    this.productSettingValues = val;
                } else {
                    this.$emit('input', val);
                }
            }
        }
    },

    methods: {
        openConfigurableOptionDocumentation() {
            jQuery(this.$refs.ipConfigHelpModal).appendTo('body').modal('show');
        },

        closeConfigurableOptionDocumentation() {
            jQuery(this.$refs.ipConfigHelpModal).modal('hide');
        },

        isDisabled(key) {
            if (this.mode !== 'productSetting') {
                return false;
            }

            return this.settingValues[key] === 'inherit';
        },

        async getProductSettings() {
            try {
                var settings = await axios.get(`${this.xhrUrl}getProductSettings&productId=${this.productId}`);
            } catch (e) {
                alert(JSON.stringify(e.response.data));
            }

            if (settings.data) {
                Vue.set(this, 'settingValues', settings.data);
            }
        },

        async saveProductSettings() {
            this.pendingXhrRequests = true;

            try {
                await axios.post(`${this.xhrUrl}saveProductSettings&productId=${this.productId}`, {
                    settings: this.settingValues
                });

                this.getProductSettings();
            } catch (e) {
                alert(JSON.stringify(e.response.data));
            } finally {
                this.pendingXhrRequests = false;
            }
        }
    }
}